/*****************************************************************************
 *
 * File:		ITC2ItemDump.cpp
 * Description:	Dumps the items within a itc2 file
 * Author:		nada
 * Copyright:	nada-labs.net
 *				The author hereby grants the use and redistribution of this
 *				code for non-commercial works only.
 * Created:		2010-01-29
 *
 ****************************************************************************/
#define _AFXDLL
#include <afx.h>
#include <windows.h>
#include <stdio.h>
#include "ITC2File.h"

void DumpITC2(const char *Filename)
{
	HANDLE	hFile, hMapping;
	DWORD	fileSizeLow, fileSizeHigh;
	ITC2Header	*ITC2File;
	ITC2Item	*Item;
	char *fileBaseName;

	//Open the file
	if((hFile = CreateFile(Filename, GENERIC_READ, 0, NULL, OPEN_EXISTING, 0, NULL)) == INVALID_HANDLE_VALUE)
	{
		unsigned int error = GetLastError();
		printf("Unable to open '%s' Error code: 0x%08X\n", Filename, error);
		return;
	}

	fileSizeLow = GetFileSize(hFile, &fileSizeHigh);

	//Map the file into memory
	if((hMapping = CreateFileMapping(hFile, NULL, PAGE_READONLY, fileSizeHigh, fileSizeLow, NULL)) == INVALID_HANDLE_VALUE)
	{
		printf("Unable to map file '%s'\n", Filename);
		CloseHandle(hFile);
		return;
	}

	if((ITC2File = (ITC2Header *)MapViewOfFile(hMapping, FILE_MAP_READ, 0, 0, 0)) == NULL)
	{
		printf("Unable to map view of file '%s'\n", Filename);
		CloseHandle(hMapping);
		CloseHandle(hFile);
		return;
	}

	//Find the relative file name
	fileBaseName = (char *)&Filename[strlen(Filename) - 1];
	while((*fileBaseName != '\\') && (fileBaseName >= Filename))
		fileBaseName--;
	fileBaseName++;

	printf("Dumping contents of %s:\n", fileBaseName);
	//dump the contents of the file
	//perform a sanity check on the contents of the header, based on what we know so far
	unsigned int HType = SWAP_ENDIAN(ITC2_TYPE_HEADER);
	if(ITC2File->SectionType == SWAP_ENDIAN(ITC2_TYPE_HEADER))
	{
		if(	(ITC2File->Unknown[0] != SWAP_ENDIAN(2)) ||
			(ITC2File->Unknown[1] != SWAP_ENDIAN(2)) ||
			(ITC2File->Unknown[2] != SWAP_ENDIAN(2)) ||
			(ITC2File->Unknown[3] != SWAP_ENDIAN(0)))
			printf("\tdifferent header flags detected!\n");

		//check that there is album artwork in this file
		if(ITC2File->Contents == SWAP_ENDIAN(ITC2_HEADER_ARTWORK))
		{
			//the file contains album artwork. lets dump it.
			unsigned int itemCount = 1;
			Item = (ITC2Item *)((unsigned long long)ITC2File + (unsigned long long)SWAP_ENDIAN(ITC2File->SectionLength));	//point to the item (64-bit compatible)

			printf("\tdumping items...\n");
			//repeat until we run out of file, the last item points past the end of the file
			while((unsigned long long)Item < ((unsigned long long)ITC2File + ((unsigned long long)fileSizeHigh << 32) + (unsigned long long)fileSizeLow))
			{
				//make sure they are item sections
				if(Item->SectionType == SWAP_ENDIAN(ITC2_TYPE_ITEM))
				{
					//verify the first lot of unknowns
					if(	(Item->Unknowns_1[0] != SWAP_ENDIAN(1)) ||
						(Item->Unknowns_1[1] != SWAP_ENDIAN(2)) ||
						(Item->Unknowns_1[2] != SWAP_ENDIAN(1)) ||
						(Item->Unknowns_1[3] != SWAP_ENDIAN(0)))
						printf("\t\t[%2i] different Unknowns_1s detected!\n", itemCount);
					if(Item->HeaderLength != SWAP_ENDIAN(0xD0))
						printf("\t\t[%2i] different header Lengths detected.\n", itemCount);

					//only know of local files and how to process them
					if(Item->ItemLocation == SWAP_ENDIAN(ITC2_ITEM_LOCATION_LOCAL))
					{
						//its a local file
						char outFilename[200];
						int offset = 0;
						char *fileExt, *fileFormat;

						//point to the image data and calculate the correct size.
						unsigned char *itemData = (unsigned char *)((unsigned long long)Item + (unsigned long long)SWAP_ENDIAN(Item->HeaderLength));
						DWORD writeSize = SWAP_ENDIAN(Item->SectionLength) - SWAP_ENDIAN(Item->HeaderLength);

						switch(SWAP_ENDIAN(Item->ItemFormat))
						{
						case ITC2_ITEM_FORMAT_PNG:
							fileExt = ".png";
							fileFormat = "PNG";
							break;

						default:
							printf("\t\t[%2i] unknown file format: %08X\n", itemCount, SWAP_ENDIAN(Item->ItemFormat));
							fileExt = ".dat";
							fileFormat = "Unknown";
							break;
						}

						//generate a filename consisting of 'trackid-itemnum-widthxheight'
						for(int i = 0; i < 8; i++)
							offset += sprintf_s(&outFilename[offset], 200 - offset, "%02X", Item->TrackId[i]);
						offset += sprintf_s(&outFilename[offset], 200 - offset, "-%02i-%ix%i%s", itemCount, SWAP_ENDIAN(Item->Width), SWAP_ENDIAN(Item->Height), fileExt);
						HANDLE hOutFile;

						//create the output file and write it to disk.
						if((hOutFile = CreateFile(outFilename, GENERIC_WRITE, 0, NULL, CREATE_ALWAYS, FILE_ATTRIBUTE_NORMAL, NULL)) != INVALID_HANDLE_VALUE)
						{
							WriteFile(hOutFile, itemData, writeSize, &writeSize, NULL);
							CloseHandle(hOutFile);
							printf("\t\t[%2i] wrote %s image to file '%s'(%i)\n", itemCount, fileFormat, outFilename, writeSize);
						} else {
							printf("\t\t[%2i] unable to create file '%s'\n", itemCount, outFilename);
						}
					} else {
						printf("\t\t[%2i] unknown file location %08X\n", itemCount, SWAP_ENDIAN(Item->ItemLocation));
					}

				} else {
					printf("\t\t[%2i] is not an item\n", itemCount);
				}

				//Go to the next item
				itemCount++;
				Item = (ITC2Item *)((unsigned long long)Item + (unsigned long long)SWAP_ENDIAN(Item->SectionLength));
			}
		}
		else
		{
			printf("\tdoesn't contain artwork\n");
		}
	}
	else
	{
		printf("\tisn't a ITC2 file\n");
	}

	UnmapViewOfFile(ITC2File);		//release the file mapping
	CloseHandle(hMapping);
	CloseHandle(hFile);

}

//ripped from MSDN and modified, FindFirstFile/FindNextFile only look in current directory.
void Recurse(LPCTSTR pstr) 
{ 
   CFileFind finder; 
	WIN32_FIND_DATA findData;
	HANDLE hFind;
 
   // build a string with wildcards 
   CString strWildcard(pstr); 
   strWildcard += _T("\\*.*"); 
 
   // start working for files 
   BOOL bWorking = finder.FindFile(strWildcard); 
 
   while (bWorking) 
   { 
      bWorking = finder.FindNextFile(); 
 
      // skip . and .. files; otherwise, we'd 
      // recur infinitely! 
 
      if (finder.IsDots()) 
         continue; 
 
      // if it's a directory, recursively search it 
      if (finder.IsDirectory()) 
      { 
         CString str = finder.GetFilePath(); 
		 CString newSearch = str + "\\*.itc2";
//		 printf("%s\n", str.GetString()); 
         
		 //try to find .itc2 files in this directory
		 if((hFind = FindFirstFile(newSearch.GetString(), &findData)) != INVALID_HANDLE_VALUE)
		{
			
			do {
				CString Filename = str + "\\" + findData.cFileName;
				DumpITC2(Filename.GetString());
			} while(FindNextFile(hFind, &findData) != 0);
			FindClose(hFind);
		}

		 Recurse(str); //search subirectories
      } 
   } 
 
   finder.Close(); 
} 


int main(int argc, char **argv)
{

	if(argc == 1)
	{
		printf("usage: %s iTunesFolder [Second iTunesFolder]...\n\tit will dump all embedded images to the current directory.", argv[0]);
		return 0;
	}

	for(int i = 1; i < argc; i++)
		Recurse(argv[i]);

	return 1;
}